	function h = LSE_FIR(Norder, Be, D)
	
	% Modification of Burrus first algorithm
	
	N = Norder+1;
	Be = Be/pi;
	h = fir3(N, Be, D);
	
	function h = fir3(N,f,m)
	
	% 	h = fir3(N,f,m) calculates an optimal least square error 
	%	multiband FIR filter from a simple lowpass design.
	%	f is a vector of break frequencies and m is a vector
	%	of ideal frequency response values at those frequencies.
	%	The format is similar to that of "remez". An example of a
	%	bandpass filter: 
 
	%	h = fir3(31,[0 0.3 0.33 0.5 0.6 1],[0 0 1 1 0 0])
	
	%	f and m must  be the same even length. m must have pairs of
	%	equal values (the ideal filter must be a pice-wise constant).
	%	Remember that an even length filter must be zero at f = 1.
	%	The multiband filter is constructed from lowpass filters
	%	designed by fir3lp.m
	
	% 	Toolbox for DIGITAL FILTERS USING MATLAB 

	% 	Report bugs to:	Wanhammar@gmail.com
	
	% 	Reference: 
	%	C.S.Burrus; Multiband Least Squares FIR Filter Design, IEEE
	%	Trans. Signal Processing, Vol. 43, No.2, pp. 412-421, Feb. 1995.
	
	L = length(f);	% Number of band edges
	if m(L) == 0
		h = zeros(1,N);	% Frequency response is zero at pi
	else
		h = [zeros(1,(N-1)/2), m(L), zeros(1, (N-1)/2)];
	end
	while L > 2
		h = h + (m(L-2)-m(L-1))*fir3lp(N,f(L-2),f(L-1)); % Construct
		L = L-2;
	end

	function  b = fir3lp(N,f1,f2,p)
	
	%	b = fir3lp(N,f1,f2,p) designs a linear phase lowpass FIR filter
	%	b(n) of length N with a least integral squared error approximation
	%	to an ideal lowpass filter with a passband from 0 to f1 and a stopband
	%	from f2 to 1. (in normalized Hertz) and a p-order spline transition 
	%	band. If p is not given, a near optimal value is calculated as 
	%	p = 0.62*N*d. 
	
	%	Reference: IEEE Trans. in SP, June 1992, page 1327.
	%	C.S. Burrus, Rice University, 8/12/92
	
	if nargin < 4, p = 0.31*N*(f2-f1); end	% Optimal spline power p
		
		w0 = pi*(f2+f1)/2;	% Average band edge
		dw = pi*(f2-f1)/2;	% Half transition width
		if rem(N,2) == 0, 
			nx  = [1/2:(N-1)/2];	% Even length index vector
		else
			nx = [1:(N-1)/2];		% Odd length index vector	
		end
		M = length(nx);
		h = (sin(w0*nx))./(pi*nx); 	%	LP filter with no transition
		if dw ~= 0 & p ~= 0			% 	p-order spline transition fn
			wd = (dw/p)*nx;			% 	Weigthing function: wt
			wt = (sin(wd)./wd).^p;
			h = wt.*h;
		end
		if rem(N,2) == 0
			b = [h(M:-1:1), h]; 	% Even length output
		else
			b = [h(M:-1:1), w0/pi, h];	%Odd length output
		end
